/*
 * Tic Tac Toe
 * demonstrates use of Mouse Input
 * includes a button, created in gimp, for reseting
 * Flashes winning squares.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;

namespace TicTacToe
{
    /// <summary>
    /// This is the main type for your game
    /// </summary>
    public class Game1 : Microsoft.Xna.Framework.Game
    {
        // Would have used the XNA type Point, but it is a struct
        // and I wanted to use a reference type to allow a null value
        class Position
        {
            public int x, y;
            public Position(int x, int y) { this.x = x; this.y = y; }
        }
        
        // Provided by default
        GraphicsDeviceManager graphics;
        SpriteBatch spriteBatch;

        const int LINE_WIDTH = 10;  // Width of the line for the board
        const int SQUARE_SIZE = 80; // Size of the square for the board
        const int RED_DURATION = 400;  // Time for blinking winning cells
        const int BOARD_SIZE = 3 * SQUARE_SIZE + 2 * LINE_WIDTH;
        const int BUTTON_HEIGHT = 40;
        const int FILL = 10;
 
        // Sprites
        Texture2D boardSprite;
        Texture2D circleSprite;
        Texture2D crossSprite;
        Texture2D redSprite;
        Texture2D resetButtonSprite;

        Texture2D[,] board = new Texture2D[3, 3]; // Board is array of sprite references

        MouseState lastMouseState = Mouse.GetState();  // Why am I tracking???  Minor optimization.
        bool oTurn = false;  // Is it O's turn?
        Position[] won = null;  // Positions that are part of the win.
        bool red = false;  // Are we currently displaying any cells red?

        int elapsedTime; // counter to track time spent blinking


        public Game1()
        {
            graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";

            // game board area is 260 x 260.  Squares are 80x80 with 10 pixel wide bars
            graphics.PreferredBackBufferWidth = BOARD_SIZE;
            graphics.PreferredBackBufferHeight = BOARD_SIZE + BUTTON_HEIGHT + 2 * FILL;

            IsMouseVisible = true;  // We want to see the mouse
        }

        /// <summary>
        /// Allows the game to perform any initialization it needs to before starting to run.
        /// This is where it can query for any required services and load any non-graphic
        /// related content.  Calling base.Initialize will enumerate through any components
        /// and initialize them as well.
        /// </summary>
        protected override void Initialize()
        {
            // TODO: Add your initialization logic here

            base.Initialize();
        }

        /// <summary>
        /// LoadContent will be called once per game and is the place to load
        /// all of your content.
        /// </summary>
        protected override void LoadContent()
        {
            // Create a new SpriteBatch, which can be used to draw textures.
            spriteBatch = new SpriteBatch(GraphicsDevice);

            // TODO: use this.Content to load your game content here
            boardSprite = Content.Load<Texture2D>(@"Sprites\tictactoeboard10x260");
            circleSprite = Content.Load<Texture2D>(@"Sprites\circle80x4");
            crossSprite = Content.Load<Texture2D>(@"Sprites\cross80x4");
            redSprite = Content.Load<Texture2D>(@"Sprites\red");
            resetButtonSprite = Content.Load<Texture2D>(@"Sprites\Reset");
        }

        /// <summary>
        /// UnloadContent will be called once per game and is the place to unload
        /// all content.
        /// </summary>
        protected override void UnloadContent()
        {
            // TODO: Unload any non ContentManager content here
        }

        /// <summary>
        /// Allows the game to run logic such as updating the world,
        /// checking for collisions, gathering input, and playing audio.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Update(GameTime gameTime)
        {
            // Allows the game to exit
            if (GamePad.GetState(PlayerIndex.One).Buttons.Back == ButtonState.Pressed)
                this.Exit();

            // TODO: Add your update logic here
            // Allow board reset
            if (Keyboard.GetState().IsKeyDown(Keys.R))
            {
                Reset();
            }

            // Win state?  (Don't test if we already know...)
            if (won == null)
            {
                won = Won();
                if (won != null)  // Someone won!
                {
                    red = true; // Want to paint the winning cells red
                    elapsedTime = RED_DURATION;  // for this amount of time.
                }
            }
            else  // Someone had already won
            {
                // Reduce time till flip of displaying red
                elapsedTime -= gameTime.ElapsedGameTime.Milliseconds;
                if (elapsedTime <= 0) {  // Time to flip state of displaying red
                    red = !red;  // Toggle whether to display winning cells as red
                    // Reset elapsed time. 
                    // Note that we *add* the duration instead of just assigning it. 
                    elapsedTime += RED_DURATION;  
                }
            }

            // Click?
            handleMouse();

            base.Update(gameTime);
        }

        /// <summary>
        /// This is called when the game should draw itself.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Draw(GameTime gameTime)
        {
            GraphicsDevice.Clear(Color.CornflowerBlue);

            // TODO: Add your drawing code here
            spriteBatch.Begin();
            spriteBatch.Draw(boardSprite, Vector2.Zero, Color.White);

            // If it is time to flash red then paint the appropriate squares red
            if (red)
            {
                for (int i = 0; i < 3; ++i)
                {
                    int row = won[i].x, col = won[i].y;
                    Vector2 pos = new Vector2(row * (SQUARE_SIZE + LINE_WIDTH), col * (SQUARE_SIZE + LINE_WIDTH));
                    spriteBatch.Draw(redSprite, pos, Color.White);
                }
            }
            // Display the X's and O's
            for (int row = 0; row < 3; ++row)
                for (int col = 0; col < 3; ++col)
                    if (board[row, col] != null)
                        spriteBatch.Draw(board[row, col],
                            new Vector2(row * (SQUARE_SIZE + LINE_WIDTH), col * (SQUARE_SIZE + LINE_WIDTH)),
                            Color.White);
            spriteBatch.Draw(resetButtonSprite,
                new Vector2((BOARD_SIZE - resetButtonSprite.Width) / 2, BOARD_SIZE + FILL), Color.White);

           
            spriteBatch.End();

            base.Draw(gameTime);
        }

        // Has anyone won?  If so, return an array of the three winning Positions.
        private Position[] Won()
        {
            // Winning row?
            for (int row = 0; row < 3; ++row)
            {
                if (board[0, row] != null
                    && board[0, row] == board[1, row]
                    && board[0, row] == board[2, row])
                {
                    return new Position[] { new Position(0, row), new Position(1, row), new Position(2, row) };
                }
            }
            // Winning column?
            for (int col = 0; col < 3; ++col)
            {
                if (board[col, 0] != null
                    && board[col, 0] == board[col, 1]
                    && board[col, 0] == board[col, 2])
                {
                    return new Position[] { new Position(col, 0), new Position(col, 1), new Position(col, 2) };
                }
            }
            // Winning diagonal top left to bottom right?
            if (board[0, 0] != null
                && board[0, 0] == board[1, 1]
                && board[0, 0] == board[2, 2])
            {
                return new Position[] { new Position(0, 0), new Position(1, 1), new Position(2, 2) };
            }
            // Winning diagonal top right to bottom left?
            if (board[0, 2] != null
                && board[0, 2] == board[1, 1]
                && board[0, 2] == board[2, 0])
            {
                return new Position[] { new Position(0, 2), new Position(1, 1), new Position(2, 0) };
            }
            return null;  // No one has won yet
        }

        // OverReset: return true iff mouse is over the reset button.
        private bool OverReset()
        {
            float xPos = Mouse.GetState().X;
            float yPos = Mouse.GetState().Y;
            return (xPos > (BOARD_SIZE - resetButtonSprite.Width) / 2)
                && (xPos < (BOARD_SIZE + resetButtonSprite.Width) / 2)
                && yPos > BOARD_SIZE + FILL
                && yPos < BOARD_SIZE + FILL + resetButtonSprite.Height;
        }

        // If the mouse is not over a square of the board, returns null
        // Otherwise returns a Position whose x and y represent the square on the board.
        private Position ComuputePosition()
        {
            float xPos = Mouse.GetState().X;
            float yPos = Mouse.GetState().Y;
            if (xPos < 0 || xPos > (3 * SQUARE_SIZE + 2 * LINE_WIDTH) ||
                yPos < 0 || yPos > (3 * SQUARE_SIZE + 2 * LINE_WIDTH))
                return null;
            int x = (int)(xPos / (SQUARE_SIZE + LINE_WIDTH));

            // Hm, have to think if these tests are redundant.
            if ((xPos - (x * (SQUARE_SIZE + LINE_WIDTH))) > SQUARE_SIZE)
            {
                return null;
            }
            int y = (int)(yPos / (SQUARE_SIZE + LINE_WIDTH));
            if ((yPos - (y * (SQUARE_SIZE + LINE_WIDTH))) > SQUARE_SIZE)
            {
                return null;
            }

            return new Position(x, y);
        }

        // Nice to have an option to reset the game rather than have to restart it
        private void Reset()
        {
            won = null;     // No one has won.
            red = false;    // and we are not painting anything red after reset.
            oTurn = false;  // We start with X's turn
            // Clear the board
            for (int row = 0; row < 3; ++row)
                for (int col = 0; col < 3; ++col)
                    board[col, row] = null;
        }

        private void handleMouse()
        {
            MouseState mouseState = Mouse.GetState();
            if (IsActive // Are we the active application?
                && mouseState.LeftButton == ButtonState.Pressed // Is the left button pressed
                && lastMouseState.LeftButton == ButtonState.Released) // And was it not pressed last time around?
            {
                if (OverReset())
                {
                    Reset();
                }
                else if (won == null) // Don't process clicks on the board if someone has won
                {
                    Position pos = ComuputePosition();  // Where did they click?
                    if (pos != null && board[pos.x, pos.y] == null)
                    {
                        // Click was in a valid posistion that wasn't occupied
                        // so set the board position to the current player's sprite
                        board[pos.x, pos.y] = (oTurn ? circleSprite : crossSprite);
                        oTurn = !oTurn;  // Toggle whose turn it is
                    }
                }
            }
            // Remember the state of the mouse.  This avoids computing positions too often.
            lastMouseState = mouseState;
        }
    }

}
